/*
ta_uart module (originally designed for MSP430G2231) provides 
half-duplex 8N1 UART implementation using TimerA Capture/Compare 
interrupt #1. UART speed is fixed at 4800bps (can be altered by 
UART_BITTIME_... constants). TimerA MUST be clocked at 32768Hz.

Example of proper configuration:

  Set an appropriate capacitor value for LFXT1 crystal:
  
  BCSCTL3=XCAP1 | XCAP0; <-Put proper value here
  
  Set 32768Hz crystal as a TimerA clock source,
  continious mode (count to 0xFFFF):
  
  TACTL=TASSEL0 | MC1;

  <Initialize RX/TX mode here>

  ...

Capture/Compare interrupt #0 is free for any purpose!

Written by YS from radiokot.ru, ysgmbx0@gmail.com

You may freely use this code any way you want,
but only at your own risk. Author gives no warranty 
and takes no responsibility on anything that could 
happen while using this code.
*/

#ifndef _TA_UART_H_
#define _TA_UART_H_

/****************************UART definitions**********************************/

//UART status/internal constants
#define UART_TX_READY             1
#define UART_STARTBIT_DETECTED    2
#define UART_RX_COMPL             4

#define UART_BITTIME_NORMAL       7
#define UART_BITTIME_FIRSTBIT     10

//UART will be implemented on this port
#define UART_PORT P1OUT
#define UART_PDIR  P1DIR
#define UART_PSEL  P1SEL

//UART RX/TX pin bitmasks
#define UART_RXPIN BIT2
#define UART_TXPIN BIT1

//These are handy characters for using with terminal
#define CH_CR 0xD
#define CH_LF 0xA

/*********************UART variables/interface functions***********************/

//UART status variable
extern volatile unsigned char UART_status;

//UART mode control functions

void UART_SetTxMode(void);
void UART_SetRxMode(void);

//UART auxillary functions

//Waits for transmission to complete
void UART_WaitTxCpl(void);

//UART TX functions

//WARNING: Using this functions inside an interrupt will
//result in infinite loop!
void UART_SendByte(unsigned char byte);
void UART_SendStr(char str[]);

//UART RX functions

//WARNING: RX-handler function is called from an interrupt,
//so it MUST be as short as possible
void UART_SetRxHandler(void (*_new_rx_handler)(unsigned char));

#endif