/*
TimerA-driven UART implementation.

For additional help, read ta_uart.h

Written by YS from radiokot.ru, ysgmbx0@gmail.com

You may freely use this code any way you want,
but only at your own risk. Author gives no warranty 
and takes no responsibility of anything that could 
happen while using this code.
*/

#include <msp430g2231.h>

#include "ta_uart.h"

volatile unsigned int UART_txbyte;
volatile unsigned char UART_rxdata;
volatile unsigned char UART_status;

void _default_rx_handler(unsigned char);

void (* volatile _uart_handler)(void);
void (* _rx_handler)(unsigned char rxbyte) = _default_rx_handler;

/*************************Internal UART handlers*******************************/

//This funcltion handles transmission while 
//UART is configured as transmitter
void _uart_tx_handler(void)
{
  static unsigned char txbitnum=0;
  
  TACCR1=(TACCR1+UART_BITTIME_NORMAL) % 65536;
  
  //If there's something to send...
  if (!(UART_status & UART_TX_READY))
  {
    //Is transmission complete?
    if (txbitnum>9)
    {
      txbitnum=0;
      UART_status|=UART_TX_READY;
    }
    else
    {
      //Sending next bit
      if (UART_txbyte & (1<<txbitnum))
        UART_PORT|=UART_TXPIN;
      else
        UART_PORT&=~UART_TXPIN;
      
      ++txbitnum;
    }
  }
}

//This is the default receive event handler
void _default_rx_handler(unsigned char ch)
{
}

//This funcltion handles reception while 
//UART is configured as a receiver
void _uart_rx_handler(void)
{
  static unsigned char rxbitnum;
  static unsigned int ta_interval;

  //If startbit was found...
  if (UART_status & UART_STARTBIT_DETECTED)
  {
    //If this is the first bit reading after
    //startbit, set timer interval to normal
    //value to read all other bits
    if (ta_interval==UART_BITTIME_FIRSTBIT)
      ta_interval=UART_BITTIME_NORMAL;
    
    TACCR1=(TACCR1+ta_interval) % 65536;
    
    //If current received bit is 'one', setting 
    //appropriate bit in byte
    if (P1IN & UART_RXPIN)
      UART_rxdata|=(1<<rxbitnum);
    
    ++rxbitnum;
    
    //All bits received?
    if (rxbitnum>7)
    {
      //Going back to waiting next start bit
      
      //Configuring RXPIN as a capture input for TimerA CC1
      P1DIR&=~UART_RXPIN;
      P1SEL|=UART_RXPIN;
  
      //Enable capture interrupt on falling 
      //edge (waiting for startbit), synchronious 
      //capture from RXPIN
      TACCTL1=CM1 | CAP | CCIE | SCS;
      
      //Setting/resetting appropriate flags
      UART_status&=~UART_STARTBIT_DETECTED;
      
      //Now received byte is completely available
      UART_status|=UART_RX_COMPL;
      
      //Calling RX handler
      _rx_handler(UART_rxdata);
      
      UART_rxdata=0;
    }
  }
  else
  {
    //Start bit found! Switching to compare mode
    //to measure bit intervals.
    TACCTL1=CCIE;
    ta_interval=UART_BITTIME_FIRSTBIT;
    TACCR1=(TACCR1+ta_interval) % 65536;
    
    //Now RXPIN will be used as GPIO
    UART_PSEL&=~UART_RXPIN;
    UART_PDIR&=~UART_RXPIN;
    
    //Initializing byte that holds receive data
    //and receive bit counter
    UART_rxdata=0;
    rxbitnum=0;
    
    //Enabling main reception procedure
    UART_status|=UART_STARTBIT_DETECTED;
  }
}

#pragma vector = TIMERA1_VECTOR

__interrupt void TA_int1_handler(void)
{
  volatile unsigned int dummy;
  
  //Dummy taiv read just to re-enable interrupt
  dummy=TAIV;
  
  //Calling actual uart function handler as set
  //by UART_SetXxMode() fns
  _uart_handler();
}

/***********************User interface functions*******************************/

//Transmitter-related:

void UART_SetTxMode(void)
{
  //Disabling UART function before
  //applying new settings
  TACCTL1=0;
  
  //Preparing port
  UART_PDIR|=UART_TXPIN;
  UART_PORT|=UART_TXPIN;
  
  //Start conditions for transmitter
  _uart_handler=_uart_tx_handler;
  UART_status=UART_TX_READY;
  
  //Capture/compare unit 1 is used for software UART
  //Setting time interval for approx. 4800bps
  TACCR1=UART_BITTIME_NORMAL;
  TACCTL1=CCIE;
}

void UART_WaitTxCpl(void)
{
  //Waiting for transmission to complete
  while (!(UART_status & UART_TX_READY));
}

void UART_SendByte(unsigned char byte)
{
  UART_WaitTxCpl();
  
  //Adding start/stop bits
  UART_txbyte=byte;
  UART_txbyte=(UART_txbyte<<1) | 0x200;
  
  UART_status&=~UART_TX_READY;
}

void UART_SendStr(char str[])
{
  int i=-1;
  
  do
  {
    ++i;
    UART_SendByte(str[i]);
  }while (str[i]!=0);
}

//Receiver-related:

void UART_SetRxMode(void)
{
  //Disabling UART function before
  //applying new settings
  TACCTL1=0;
  
  //RXPIN is capture input for TimerA CC1
  P1DIR&=~UART_RXPIN;
  P1SEL|=UART_RXPIN;
  
  //Start conditions for receiver
  _uart_handler=_uart_rx_handler;
  UART_status=0;
  
  //Enable capture interrupt on falling 
  //edge (waiting for startbit), synchronious 
  //capture from P1.2 (RXD)
  TACCTL1=CM1 | CAP | CCIE | SCS;
}

void UART_SetRxHandler(void (*_new_rx_handler)(unsigned char))
{
  _rx_handler=_new_rx_handler;
}